<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\Layout
 * @author  StudioPress
 * @license GPL-2.0-or-later
 * @link    https://my.studiopress.com/themes/genesis/
 */

add_filter( 'content_width', 'genesis_content_width', 10, 3 );
/**
 * Filter the content width based on the user selected layout.
 *
 * @since 1.6.0
 *
 * @param int $default Default width.
 * @param int $small   Small width.
 * @param int $large   Large width.
 * @return int Content width.
 */
function genesis_content_width( $default, $small, $large ) {

	switch ( genesis_site_layout( 0 ) ) {
		case 'full-width-content':
			$width = $large;
			break;
		case 'content-sidebar-sidebar':
		case 'sidebar-content-sidebar':
		case 'sidebar-sidebar-content':
			$width = $small;
			break;
		default:
			$width = $default;
	}

	return $width;

}

add_filter( 'body_class', 'genesis_custom_body_class', 15 );
/**
 * Add custom field body class(es) to the body classes.
 *
 * It accepts values from a per-post or per-page custom field, and only outputs when viewing a singular page.
 *
 * @since 1.4.0
 *
 * @param array $classes Existing body classes.
 * @return array Amended body classes.
 */
function genesis_custom_body_class( array $classes ) {

	$new_class = is_singular() ? genesis_get_custom_field( '_genesis_custom_body_class' ) : null;

	if ( $new_class ) {
		$classes[] = $new_class;
	}

	return $classes;

}

add_filter( 'body_class', 'genesis_header_body_classes' );
/**
 * Add header-* classes to the body class.
 *
 * We can use pseudo-variables in our CSS file, which helps us achieve multiple header layouts with minimal code.
 *
 * @since 1.0.0
 *
 * @param array $classes Existing body classes.
 * @return array Amended body classes.
 */
function genesis_header_body_classes( array $classes ) {

	if ( current_theme_supports( 'custom-header' ) ) {
		if ( get_theme_support( 'custom-header', 'default-text-color' ) !== get_header_textcolor() || get_theme_support( 'custom-header', 'default-image' ) !== get_header_image() ) {
			$classes[] = 'custom-header';
		}
	}

	if ( 'image' === genesis_get_option( 'blog_title' ) || ( get_header_image() && ! display_header_text() ) ) {
		$classes[] = 'header-image';
	}

	if ( ! is_active_sidebar( 'header-right' ) && ! has_action( 'genesis_header_right' ) ) {
		$classes[] = 'header-full-width';
	}

	return $classes;

}

add_filter( 'body_class', 'genesis_layout_body_classes' );
/**
 * Add site layout classes to the body classes.
 *
 * We can use pseudo-variables in our CSS file, which helps us achieve multiple site layouts with minimal code.
 *
 * @since 1.0.0
 *
 * @param array $classes Existing body classes.
 * @return array Amended body classes.
 */
function genesis_layout_body_classes( array $classes ) {

	$site_layout = genesis_site_layout();

	if ( $site_layout ) {
		$classes[] = $site_layout;
	}

	return $classes;

}

add_filter( 'body_class', 'genesis_title_hidden_body_class' );
/**
 * Adds a `genesis-title-hidden` body class if title output is suppressed on the current page.
 *
 * @since 3.1.0
 *
 * @param array $classes Existing body classes.
 * @return array Amended body classes.
 */
function genesis_title_hidden_body_class( array $classes ) {

	if ( genesis_entry_header_hidden_on_current_page() ) {
		$classes[] = 'genesis-title-hidden';
	}

	return $classes;

}

add_filter( 'body_class', 'genesis_breadcrumbs_hidden_body_class' );
/**
 * Adds a `genesis-breadcrumbs-hidden` body class if breadcrumbs are hidden on the current page.
 *
 * @since 3.1.0
 *
 * @param array $classes Existing body classes.
 * @return array Amended body classes.
 */
function genesis_breadcrumbs_hidden_body_class( array $classes ) {

	if ( genesis_breadcrumbs_hidden_on_current_page() || genesis_breadcrumbs_disabled_on_current_page() ) {
		$classes[] = 'genesis-breadcrumbs-hidden';
		return $classes;
	}

	$classes[] = 'genesis-breadcrumbs-visible';

	return $classes;

}

add_filter( 'body_class', 'genesis_singular_image_hidden_body_class' );
/**
 * Adds a `genesis-singular-image-hidden` body class if “hide featured image” is enabled.
 *
 * @since 3.1.0
 *
 * @param array $classes Existing body classes.
 * @return array Amended body classes.
 */
function genesis_singular_image_hidden_body_class( array $classes ) {

	if ( ! is_singular() ) {
		return $classes;
	}

	if ( ! post_type_supports( get_post_type(), 'genesis-singular-images' ) ) {
		return $classes;
	}

	if ( genesis_singular_image_hidden_on_current_page() ) {
		$classes[] = 'genesis-singular-image-hidden';
	}

	return $classes;

}

add_filter( 'body_class', 'genesis_singular_image_visible_body_class' );
/**
 * Adds a `genesis-singular-image-visible` body class.
 *
 * @since 3.1.1
 *
 * @param array $classes Existing body classes.
 * @return array Amended body classes.
 */
function genesis_singular_image_visible_body_class( array $classes ) {

	if (
		is_singular()
		&& ! genesis_singular_image_hidden_on_current_page()
		&& genesis_get_singular_image()
	) {
		$classes[] = 'genesis-singular-image-visible';
	}

	return $classes;

}

add_filter( 'body_class', 'genesis_footer_widgets_hidden_body_class' );
/**
 * Adds `genesis-footer-widgets-hidden` and genesis-footer-widgets-visible` body classes if footer widgets
 * are supported by the child theme and the Footer Widgets Panel is available in the Genesis Sidebar.
 *
 * @since 3.2.0
 *
 * @param array $classes Existing body classes.
 * @return array Amended body classes.
 */
function genesis_footer_widgets_hidden_body_class( array $classes ) {

	$footer_widgets                = get_theme_support( 'genesis-footer-widgets' );
	$footer_widgets_toggle_enabled = apply_filters( 'genesis_footer_widgets_toggle_enabled', true );

	// Return if theme does not support footer widgets or the Footer Widgets sidebar panel is disabled.
	if ( ! $footer_widgets || ! isset( $footer_widgets[0] ) || ! is_numeric( $footer_widgets[0] ) || ! $footer_widgets_toggle_enabled ) {
		return $classes;
	}

	if ( ! is_active_sidebar( 'footer-1' ) || genesis_footer_widgets_hidden_on_current_page() ) {
		$classes[] = 'genesis-footer-widgets-hidden';
		return $classes;
	}

	$classes[] = 'genesis-footer-widgets-visible';

	return $classes;

}

add_filter( 'body_class', 'genesis_archive_no_results_body_class' );
/**
 * Add archive-no-results body class on empty archive pages.
 *
 * Allows CSS styling of archive pages that have no results.
 *
 * @since 2.2.0
 *
 * @global WP_Query $wp_query Query object.
 *
 * @param array $classes Existing body classes.
 * @return array Amended body classes.
 */
function genesis_archive_no_results_body_class( array $classes ) {

	global $wp_query;

	if ( ! $wp_query->posts && is_archive() ) {
		$classes[] = 'archive-no-results';
	}

	return $classes;

}

add_filter( 'body_class', 'genesis_style_selector_body_classes' );
/**
 * Add style selector classes to the body classes.
 *
 * Enables style selector support in child themes, which helps us achieve multiple site styles with minimal code.
 *
 * @since 1.8.0
 *
 * @param array $classes Existing body classes.
 * @return array Amended body classes.
 */
function genesis_style_selector_body_classes( array $classes ) {

	$current = genesis_get_option( 'style_selection' );

	if ( $current ) {
		$classes[] = $current;
	}

	return $classes;

}

add_filter( 'body_class', 'genesis_cpt_archive_body_class', 15 );
/**
 * Adds a custom class to the custom post type archive body classes.
 *
 * It accepts a value from the archive settings page.
 *
 * @since 2.0.0
 *
 * @param array $classes Existing body classes.
 * @return array Amended body classes.
 */
function genesis_cpt_archive_body_class( array $classes ) {

	if ( ! is_post_type_archive() || ! genesis_has_post_type_archive_support() ) {
		return $classes;
	}

	$new_class = genesis_get_cpt_option( 'body_class' );

	if ( $new_class ) {
		$classes[] = $new_class;
	}

	return $classes;

}

add_action( 'genesis_after_content', 'genesis_get_sidebar' );
/**
 * Output the sidebar.php file if layout allows for it.
 *
 * @since 1.0.0
 */
function genesis_get_sidebar() {

	$site_layout = genesis_site_layout();

	// Don't load sidebar on pages that don't need it.
	if ( 'full-width-content' === $site_layout ) {
		return;
	}

	get_sidebar();

}

add_action( 'genesis_after_content_sidebar_wrap', 'genesis_get_sidebar_alt' );
/**
 * Output the sidebar_alt.php file if layout allows for it.
 *
 * @since 1.0.0
 */
function genesis_get_sidebar_alt() {

	$site_layout = genesis_site_layout();

	// Don't load sidebar-alt on pages that don't need it.
	if ( in_array( $site_layout, [ 'content-sidebar', 'sidebar-content', 'full-width-content' ], true ) ) {
		return;
	}

	get_sidebar( 'alt' );

}
